import { ErrorCode, InvalidPayloadError, isDirectusError } from '@directus/errors';
import express from 'express';
import { assign } from 'lodash-es';
import { respond } from '../middleware/respond.js';
import useCollection from '../middleware/use-collection.js';
import { validateBatch } from '../middleware/validate-batch.js';
import { MetaService } from '../services/meta.js';
import { VersionsService } from '../services/versions.js';
import asyncHandler from '../utils/async-handler.js';
import { sanitizeQuery } from '../utils/sanitize-query.js';
const router = express.Router();
router.use(useCollection('directus_versions'));
router.post('/', asyncHandler(async (req, res, next) => {
    const service = new VersionsService({
        accountability: req.accountability,
        schema: req.schema,
    });
    const savedKeys = [];
    if (Array.isArray(req.body)) {
        const keys = await service.createMany(req.body);
        savedKeys.push(...keys);
    }
    else {
        const primaryKey = await service.createOne(req.body);
        savedKeys.push(primaryKey);
    }
    try {
        if (Array.isArray(req.body)) {
            const records = await service.readMany(savedKeys, req.sanitizedQuery);
            res.locals['payload'] = { data: records };
        }
        else {
            const record = await service.readOne(savedKeys[0], req.sanitizedQuery);
            res.locals['payload'] = { data: record };
        }
    }
    catch (error) {
        if (isDirectusError(error, ErrorCode.Forbidden)) {
            return next();
        }
        throw error;
    }
    return next();
}), respond);
const readHandler = asyncHandler(async (req, res, next) => {
    const service = new VersionsService({
        accountability: req.accountability,
        schema: req.schema,
    });
    const metaService = new MetaService({
        accountability: req.accountability,
        schema: req.schema,
    });
    let result;
    if (req.singleton) {
        result = await service.readSingleton(req.sanitizedQuery);
    }
    else if (req.body.keys) {
        result = await service.readMany(req.body.keys, req.sanitizedQuery);
    }
    else {
        result = await service.readByQuery(req.sanitizedQuery);
    }
    const meta = await metaService.getMetaForQuery(req.collection, req.sanitizedQuery);
    res.locals['payload'] = { data: result, meta };
    return next();
});
router.get('/', validateBatch('read'), readHandler, respond);
router.search('/', validateBatch('read'), readHandler, respond);
router.get('/:pk', asyncHandler(async (req, res, next) => {
    const service = new VersionsService({
        accountability: req.accountability,
        schema: req.schema,
    });
    const record = await service.readOne(req.params['pk'], req.sanitizedQuery);
    res.locals['payload'] = { data: record || null };
    return next();
}), respond);
router.patch('/', validateBatch('update'), asyncHandler(async (req, res, next) => {
    const service = new VersionsService({
        accountability: req.accountability,
        schema: req.schema,
    });
    let keys = [];
    if (Array.isArray(req.body)) {
        keys = await service.updateBatch(req.body);
    }
    else if (req.body.keys) {
        keys = await service.updateMany(req.body.keys, req.body.data);
    }
    else {
        const sanitizedQuery = sanitizeQuery(req.body.query, req.accountability);
        keys = await service.updateByQuery(sanitizedQuery, req.body.data);
    }
    try {
        const result = await service.readMany(keys, req.sanitizedQuery);
        res.locals['payload'] = { data: result || null };
    }
    catch (error) {
        if (isDirectusError(error, ErrorCode.Forbidden)) {
            return next();
        }
        throw error;
    }
    return next();
}), respond);
router.patch('/:pk', asyncHandler(async (req, res, next) => {
    const service = new VersionsService({
        accountability: req.accountability,
        schema: req.schema,
    });
    const primaryKey = await service.updateOne(req.params['pk'], req.body);
    try {
        const record = await service.readOne(primaryKey, req.sanitizedQuery);
        res.locals['payload'] = { data: record || null };
    }
    catch (error) {
        if (isDirectusError(error, ErrorCode.Forbidden)) {
            return next();
        }
        throw error;
    }
    return next();
}), respond);
router.delete('/', validateBatch('delete'), asyncHandler(async (req, _res, next) => {
    const service = new VersionsService({
        accountability: req.accountability,
        schema: req.schema,
    });
    if (Array.isArray(req.body)) {
        await service.deleteMany(req.body);
    }
    else if (req.body.keys) {
        await service.deleteMany(req.body.keys);
    }
    else {
        const sanitizedQuery = sanitizeQuery(req.body.query, req.accountability);
        await service.deleteByQuery(sanitizedQuery);
    }
    return next();
}), respond);
router.delete('/:pk', asyncHandler(async (req, _res, next) => {
    const service = new VersionsService({
        accountability: req.accountability,
        schema: req.schema,
    });
    await service.deleteOne(req.params['pk']);
    return next();
}), respond);
router.get('/:pk/compare', asyncHandler(async (req, res, next) => {
    const service = new VersionsService({
        accountability: req.accountability,
        schema: req.schema,
    });
    const version = await service.readOne(req.params['pk']);
    const { outdated, mainHash } = await service.verifyHash(version['collection'], version['item'], version['hash']);
    const saves = await service.getVersionSavesById(version['id']);
    const current = assign({}, ...saves);
    const main = await service.getMainItem(version['collection'], version['item']);
    res.locals['payload'] = { data: { outdated, mainHash, current, main } };
    return next();
}), respond);
router.post('/:pk/save', asyncHandler(async (req, res, next) => {
    const service = new VersionsService({
        accountability: req.accountability,
        schema: req.schema,
    });
    const version = await service.readOne(req.params['pk']);
    const mainItem = await service.getMainItem(version['collection'], version['item']);
    await service.save(req.params['pk'], req.body);
    const saves = await service.getVersionSavesById(req.params['pk']);
    const result = assign(mainItem, ...saves);
    res.locals['payload'] = { data: result || null };
    return next();
}), respond);
router.post('/:pk/promote', asyncHandler(async (req, res, next) => {
    if (typeof req.body.mainHash !== 'string') {
        throw new InvalidPayloadError({ reason: `"mainHash" field is required` });
    }
    const service = new VersionsService({
        accountability: req.accountability,
        schema: req.schema,
    });
    const updatedItemKey = await service.promote(req.params['pk'], req.body.mainHash, req.body?.['fields']);
    res.locals['payload'] = { data: updatedItemKey || null };
    return next();
}), respond);
export default router;
