"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.connectInParallel = connectInParallel;
exports.connectInSequence = connectInSequence;
exports.lookupAllAddresses = lookupAllAddresses;
var _net = _interopRequireDefault(require("net"));
var _nodeUrl = _interopRequireDefault(require("node:url"));
var _abortError = _interopRequireDefault(require("./errors/abort-error"));
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
async function connectInParallel(options, lookup, signal) {
  if (signal.aborted) {
    throw new _abortError.default();
  }
  const addresses = await lookupAllAddresses(options.host, lookup, signal);
  return await new Promise((resolve, reject) => {
    const sockets = new Array(addresses.length);
    const errors = [];
    function onError(err) {
      errors.push(err);
      this.removeListener('error', onError);
      this.removeListener('connect', onConnect);
      this.destroy();
      if (errors.length === addresses.length) {
        signal.removeEventListener('abort', onAbort);
        reject(new AggregateError(errors, 'Could not connect (parallel)'));
      }
    }
    function onConnect() {
      signal.removeEventListener('abort', onAbort);
      for (let j = 0; j < sockets.length; j++) {
        const socket = sockets[j];
        if (this === socket) {
          continue;
        }
        socket.removeListener('error', onError);
        socket.removeListener('connect', onConnect);
        socket.destroy();
      }
      resolve(this);
    }
    const onAbort = () => {
      for (let j = 0; j < sockets.length; j++) {
        const socket = sockets[j];
        socket.removeListener('error', onError);
        socket.removeListener('connect', onConnect);
        socket.destroy();
      }
      reject(new _abortError.default());
    };
    for (let i = 0, len = addresses.length; i < len; i++) {
      const socket = sockets[i] = _net.default.connect({
        ...options,
        host: addresses[i].address,
        family: addresses[i].family
      });
      socket.on('error', onError);
      socket.on('connect', onConnect);
    }
    signal.addEventListener('abort', onAbort, {
      once: true
    });
  });
}
async function connectInSequence(options, lookup, signal) {
  if (signal.aborted) {
    throw new _abortError.default();
  }
  const errors = [];
  const addresses = await lookupAllAddresses(options.host, lookup, signal);
  for (const address of addresses) {
    try {
      return await new Promise((resolve, reject) => {
        const socket = _net.default.connect({
          ...options,
          host: address.address,
          family: address.family
        });
        const onAbort = () => {
          socket.removeListener('error', onError);
          socket.removeListener('connect', onConnect);
          socket.destroy();
          reject(new _abortError.default());
        };
        const onError = err => {
          signal.removeEventListener('abort', onAbort);
          socket.removeListener('error', onError);
          socket.removeListener('connect', onConnect);
          socket.destroy();
          reject(err);
        };
        const onConnect = () => {
          signal.removeEventListener('abort', onAbort);
          socket.removeListener('error', onError);
          socket.removeListener('connect', onConnect);
          resolve(socket);
        };
        signal.addEventListener('abort', onAbort, {
          once: true
        });
        socket.on('error', onError);
        socket.on('connect', onConnect);
      });
    } catch (err) {
      if (err instanceof Error && err.name === 'AbortError') {
        throw err;
      }
      errors.push(err);
      continue;
    }
  }
  throw new AggregateError(errors, 'Could not connect (sequence)');
}

/**
 * Look up all addresses for the given hostname.
 */
async function lookupAllAddresses(host, lookup, signal) {
  if (signal.aborted) {
    throw new _abortError.default();
  }
  if (_net.default.isIPv6(host)) {
    return [{
      address: host,
      family: 6
    }];
  } else if (_net.default.isIPv4(host)) {
    return [{
      address: host,
      family: 4
    }];
  } else {
    return await new Promise((resolve, reject) => {
      const onAbort = () => {
        reject(new _abortError.default());
      };
      signal.addEventListener('abort', onAbort);
      lookup(_nodeUrl.default.domainToASCII(host), {
        all: true
      }, (err, addresses) => {
        signal.removeEventListener('abort', onAbort);
        err ? reject(err) : resolve(addresses);
      });
    });
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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