import jwt from 'jsonwebtoken'
import Stripe from 'stripe'
import { OAuth2Client } from 'google-auth-library'
import jwksClient from 'jwks-rsa'

// Create JWKS client
const client = jwksClient({
  jwksUri: 'https://appleid.apple.com/auth/keys'
})

// Get Apple’s signing key
function getAppleSigningKey (kid) {
  return new Promise((resolve, reject) => {
    client.getSigningKey(kid, (err, key) => {
      if (err) return reject(err)
      const signingKey = key.getPublicKey()
      resolve(signingKey)
    })
  })
}
// Verify Apple ID token
async function verifyAppleToken (idToken) {
  const decodedHeader = jwt.decode(idToken, { complete: true })
  const kid = decodedHeader.header.kid
  const alg = decodedHeader.header.alg
  const signingKey = await getAppleSigningKey(kid)
  const payload = jwt.verify(idToken, signingKey, {
    algorithms: [alg]
  })
  return payload
}

export default {
  id: 'api',
  handler: (router, context) => {
    const { services, database, getSchema, env, logger, emitter } = context
    const {
      ItemsService,
      UsersService,
      RolesService,
      PermissionsService,
      AuthenticationService
    } = services

    function areAllFieldsPresent (requiredFields, obj) {
      return requiredFields.every(field => field in obj)
    }
    function getDaysBetween (startDateStr, endDateStr) {
      const start = new Date(startDateStr)
      const end = new Date(endDateStr)

      // Get the difference in milliseconds
      const diffMs = end - start

      // Convert milliseconds to days
      const diffDays = diffMs / (1000 * 60 * 60 * 24)

      return Math.round(diffDays)
    }

    // sso login
    {
      const GOOGLE_CLIENT_IDS = [
        '431467211911-6i0i4pqo07u884d8tu6p3q8rijmrh1jr.apps.googleusercontent.com'
      ]

      const JWT_SECRET = '1Fa0WrHz2uK_TM9YSS4Cp4JEfiG5kTjt'
      const googleClient = new OAuth2Client()
      const CUSTOMER_ROLE_ID = 'c55e28e3-9333-426e-8a5d-9a1da45afe5e'

      router.post('/google-login', async (req, res) => {
        try {
          const { idToken } = req.body
          if (!idToken) {
            return res.status(400).json({ error: 'ID token is required' })
          }

          // Verify Google ID token
          const ticket = await googleClient.verifyIdToken({
            idToken,
            audience: GOOGLE_CLIENT_IDS
          })
          const payload = ticket.getPayload()
          const { email, name, picture, sub: googleId } = payload

          const schema = await getSchema()
          const usersService = new UsersService({
            schema,
            accountability: null
          })

          // Check if user exists
          let user = await database('directus_users').where({ email }).first()

          const tokenPayload = {
            email: email,
            role: CUSTOMER_ROLE_ID,
            iat: Math.floor(Date.now() / 1000), // Issued at time
            exp: Math.floor(Date.now() / 1000) + 60 * 60 * 3 // Expiration time (3 hours from now)
          }

          const token = jwt.sign(tokenPayload, JWT_SECRET)
          const parts = name.trim().split(/\s+/) // Splits by any number of spaces
          const first_name = parts[0] || ''
          const last_name = parts.length > 1 ? parts.slice(1).join(' ') : ''

          if (!user) {
            // Create new user
            user = await usersService.createOne({
              email,
              first_name,
              last_name,
              avatar_url: picture,
              external_identifier: email,
              status: 'active',
              role: CUSTOMER_ROLE_ID,
              provider: 'google',
              token: token
            })
          } else {
            if (user.provider != 'google' && user.provider != 'Google') {
              return res.status(300).json({
                error:
                  'This email is already registered, please login using your email and password.'
              })
            }
            // Update user
            user = await usersService.updateOne(user.id, {
              first_name,
              last_name,
              token
            })
          }

          return res.json({
            user,
            access_token: token
          })
        } catch (error) {
          console.error(error)
          res
            .status(500)
            .json({ error: 'Authentication failed', message: error.message })
        }
      })

      router.post('/apple-login', async (req, res) => {
        const schema = await getSchema()
        const usersService = new UsersService({ schema, accountability: null })

        const {
          identityToken,
          authorizationCode,
          userIdentifier,
          first_name,
          last_name
        } = req.body

        try {
          const payload = await verifyAppleToken(identityToken)
          console.log(payload)
          const appleUserId = payload.sub // Apple’s unique user ID
          const email = payload.email
          const emailVerified = payload.email_verified

          const token = jwt.sign(userIdentifier, JWT_SECRET)
          // Store or retrieve user from your database
          let user = await database('directus_users').where({ email }).first()
          // let user = await db.findUserByAppleId(appleUserId);

          if (!user) {
            // Create new user
            user = await usersService.createOne({
              email,
              first_name: first_name,
              last_name: last_name,
              // avatar_url: picture,
              external_identifier: email,
              status: 'active',
              role: CUSTOMER_ROLE_ID,
              provider: 'apple',
              token: token
            })
          } else {
            if (user.provider != 'apple' && user.provider != 'Apple') {
              return res.status(300).json({
                error:
                  'This email is already registered, please login using your email and password.'
              })
            }
            // Update user
            user = await usersService.updateOne(user.id, {
              // first_name: first_name,
              // last_name: last_name,
              token
            })
          }
          // Generate session/token
          return res.json({
            user,
            access_token: token
          })
        } catch (error) {
          console.error('Apple sign-in error:', error)
          res.status(401).json({ error: 'Invalid Apple identity token' })
        }
      })
    }
    // stripe related routes
    {
      // const STRIPE_SECRET_KEY='sk_test_51RpmrY2LjmNyxgpay1VHu7fx9nTkvodQuyIg8TegWI0pXM5h6TTJEnNqe2xJJIuZWjy98c6QEKIJK4LHiEmPJD0y00JakkSVH0'
      const STRIPE_WEBHOOK_SECRET = ''
      const stripe = new Stripe(env.STRIPE_SECRET_KEY)

      async function safelyAttachPaymentMethod (customerId, paymentMethodId) {
        try {
          const pm = await stripe.paymentMethods.retrieve(paymentMethodId)
          // Already attached to this customer
          if (pm.customer === customerId) {
            return {
              success: true,
              alreadyAttached: true,
              paymentMethod: pm
            }
          }
          // Attached to another customer
          if (pm.customer) {
            return {
              success: false,
              error: 'Payment method belongs to another account',
              code: 'payment_method_attached_to_other'
            }
          }
          // Perform attachment
          const attachedPm = await stripe.paymentMethods.attach(
            paymentMethodId,
            {
              customer: customerId
            }
          )
          return {
            success: true,
            alreadyAttached: false,
            paymentMethod: attachedPm
          }
        } catch (error) {
          console.error('Payment method attachment error:', error)

          // Handle specific Stripe error codes
          const errorMap = {
            payment_method_already_attached:
              'Payment method already attached to another customer',
            resource_missing: 'Invalid payment method ID',
            parameter_unknown: 'Invalid customer ID'
          }

          return {
            success: false,
            error: errorMap[error.code] || error.message,
            code: error.code
          }
        }
      }

      async function handleSubscriptionUpdated (subscription) {
        // Calculate period dates (handles timezone conversions)
        const current_period_end = new Date(
          subscription.current_period_end * 1000
        )
        const current_period_start = new Date(
          subscription.current_period_start * 1000
        )

        await directus.items('subscription').updateByQuery({
          filter: { stripe_sub_id: { _eq: subscription.id } },
          data: {
            status: subscription.status,
            current_period_start: current_period_start,
            current_period_end: current_period_end,
            cancel_at_period_end: subscription.cancel_at_period_end || false
          }
        })
      }

      async function handleInvoicePaid (invoice) {
        const subscriptionId = invoice.subscription

        await directus.items('subscription').updateByQuery({
          filter: { stripe_sub_id: { _eq: subscriptionId } },
          data: {
            status: 'active',
            // Period dates already set during creation
            last_payment_at: new Date()
          }
        })
      }

      async function handlePaymentFailed (invoice) {
        try {
          const subscriptionId = invoice.subscription
          const customerId = invoice.customer
          const paymentIntent = invoice.payment_intent

          // 1. Update subscription status in your database
          await directus.items('subscription').updateByQuery({
            filter: { stripe_sub_id: { _eq: subscriptionId } },
            data: {
              status: 'past_due',
              last_payment_failed_at: new Date(),
              failure_reason:
                invoice.attempt_count > 1
                  ? 'recurring_payment_failed'
                  : 'initial_payment_failed'
            }
          })

          // 2. Retrieve customer details
          const customer = await stripe.customers.retrieve(customerId)
          const user = await directus
            .items('directus_users')
            .readOne(customer.metadata.directus_user_id)

          // 3. Check for alternative payment methods
          const paymentMethods = await stripe.paymentMethods.list({
            customer: customerId,
            type: 'card'
          })

          // 4. Attempt recovery if alternative methods exist
          if (paymentMethods.data.length > 1) {
            const otherMethods = paymentMethods.data.filter(
              pm => pm.id !== invoice.payment_method
            )

            for (const method of otherMethods) {
              try {
                // Retry payment with alternate method
                const updatedInvoice = await stripe.invoices.pay(invoice.id, {
                  payment_method: method.id
                })

                if (updatedInvoice.status === 'paid') {
                  await handleInvoicePaid(updatedInvoice) // Reuse your success handler
                  return
                }
              } catch (retryError) {
                console.error(
                  `Retry failed with method ${method.id}:`,
                  retryError
                )
              }
            }
          }

          // 5. Notify customer (email + in-app)
          await sendPaymentFailureNotification({
            email: customer.email,
            name: user.first_name,
            amount: invoice.amount_due / 100,
            currency: invoice.currency,
            invoice_url: invoice.hosted_invoice_url,
            retry_attempts: invoice.attempt_count,
            next_retry: new Date(invoice.next_payment_attempt * 1000)
          })

          // 6. Log the failure for analytics
          await directus.items('payment_failures').createOne({
            user: customer.metadata.directus_user_id,
            subscription: subscriptionId,
            invoice_id: invoice.id,
            amount: invoice.amount_due,
            failure_reason:
              invoice.payment_intent?.last_payment_error?.message || 'unknown',
            payment_method: invoice.payment_method
          })
        } catch (error) {
          console.error('Error handling payment failure:', error)
          // Implement your error reporting here
        }
      }

      async function sendPaymentFailureNotification (params) {
        // 1. Send email
        await emailService.send({
          to: params.email,
          template: 'payment-failed',
          data: {
            name: params.name,
            amount: `${params.currency.toUpperCase()} ${params.amount.toFixed(
              2
            )}`,
            invoice_url: params.invoice_url,
            next_retry: params.next_retry.toLocaleDateString(),
            support_email: 'support@raha.app'
          }
        })

        // 2. Create in-app notification
        await directus.items('notifications').createOne({
          user: params.user_id,
          type: 'payment_failed',
          message: `Payment failed for ${params.currency}${params.amount}`,
          action_url: `/account/billing?invoice=${params.invoice_id}`,
          read: false
        })
      }

      function normalizePaymentMethod (pm, isDefault = false) {
        return {
          id: pm.id,
          type: pm.type,
          isDefault,
          created: new Date(pm.created * 1000),
          card: pm.card
            ? {
                brand: pm.card.brand,
                last4: pm.card.last4,
                exp_month: pm.card.exp_month,
                exp_year: pm.card.exp_year,
                country: pm.card.country
              }
            : null,
          billing_details: pm.billing_details
        }
      }

      router.post('/add-stripe-customer', async (req, res) => {
        const userService = new ItemsService('directus_users', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        try {
          let userId = req.accountability.user
          const user = await userService.readOne(userId)
          if (user.stripe_customer_id) {
            const customer = await stripe.customers.update(
              user.stripe_customer_id,
              {
                address: '',
                description: 'Raha app',
                name: `${user.first_name} ${user.last_name}`,
                phone: user.phone_number,
                email: user.email,
                metadata: {
                  directus_user_id: userId,
                  created_via: 'add_stripe_customer'
                }
              }
            )
            return res.json({
              success: true,
              customer: customer
            })
          } else {
            const customer = await stripe.customers.create({
              address: '',
              description: 'Raha app',
              name: `${user.first_name} ${user.last_name}`,
              phone: user.phone_number,
              email: user.email
            })

            await userService.updateOne(userId, {
              stripe_customer_id: customer.id
            })
            return res.json({
              success: true,
              customer: customer
            })
          }
        } catch (error) {
          return res.json({
            success: false,
            error: error.message
          })
        }
      })

      router.post('/save-payment-method', async (req, res) => {
        const userService = new ItemsService('directus_users', {
          schema: req.schema,
          accountability: req.accountability
        })

        try {
          // 1. Validate request
          if (!req.body.paymentMethodId) {
            return res
              .status(400)
              .json({ error: 'paymentMethodId is required' })
          }

          const userId = req.accountability.user
          const user = await userService.readOne(userId)

          // 2. Create Stripe customer if doesn't exist
          let customerId = user.stripe_customer_id
          if (!customerId) {
            const customer = await stripe.customers.create({
              email: user.email,
              name: `${user.first_name} ${user.last_name}`,
              metadata: {
                directus_user_id: userId,
                created_via: 'payment_method_attachment'
              },
              phone: user.phone // Optional
            })

            customerId = customer.id

            // Update user record with new customer ID
            await userService.updateOne(userId, {
              stripe_customer_id: customerId
            })
          }

          // 3. Attach payment method
          const { success, error, paymentMethod } =
            await safelyAttachPaymentMethod(
              customerId,
              req.body.paymentMethodId
            )

          if (!success) {
            return res.status(400).json({ error })
          }

          // 4. Set as default if requested (default true)
          const setAsDefault = req.body.setAsDefault !== false
          if (setAsDefault) {
            await stripe.customers.update(customerId, {
              invoice_settings: {
                default_payment_method: req.body.paymentMethodId
              }
            })
          }

          // 5. Return complete payment method details
          res.json({
            success: true,
            customerCreated: !user.stripe_customer_id, // Indicates if new customer was made
            paymentMethod: normalizePaymentMethod(paymentMethod, setAsDefault)
          })
        } catch (error) {
          console.error('Payment method save error:', error)

          // Handle Stripe rate limiting
          if (error.type === 'StripeRateLimitError') {
            return res.status(429).json({
              error: 'Too many requests. Please try again later.'
            })
          }

          res.status(500).json({
            error: error.message
          })
        }
      })

      router.post('/detach-payment-method', async (req, res) => {
        const userService = new ItemsService('directus_users', {
          schema: req.schema,
          accountability: req.accountability
        })

        try {
          // 1. Validate request
          if (!req.body.paymentMethodId) {
            return res
              .status(400)
              .json({ error: 'paymentMethodId is required' })
          }

          const userId = req.accountability.user
          const user = await userService.readOne(userId)

          // 2. Create Stripe customer if doesn't exist
          let customerId = user.stripe_customer_id

          const paymentMethod = await stripe.paymentMethods.detach(
            req.body.paymentMethodId
          )

          // 5. Return complete payment method details
          res.json({
            success: true
          })
        } catch (error) {
          res.status(500).json({
            error: error.message
          })
        }
      })

      router.get('/payment-methods', async (req, res) => {
        const userService = new ItemsService('directus_users', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        try {
          let userId = req.accountability.user
          const user = await userService.readOne(userId)

          const paymentMethods = await stripe.paymentMethods.list({
            customer: user.stripe_customer_id,
            type: 'card'
          })

          res.json({
            success: true,
            paymentMethods: paymentMethods.data.map(pm => ({
              id: pm.id,
              brand: pm.card.brand,
              last4: pm.card.last4,
              expMonth: pm.card.exp_month,
              expYear: pm.card.exp_year,
              isDefault: pm.id === req.query.defaultPaymentMethod
            }))
          })
        } catch (error) {
          res.status(400).json({ error: error.message })
        }
      })

      router.post('/set-default-method', async (req, res) => {
        const userService = new ItemsService('directus_users', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        try {
          let userId = req.accountability.user
          const user = await userService.readOne(userId)

          await stripe.customers.update(user.stripe_customer_id, {
            invoice_settings: {
              default_payment_method: req.body.paymentMethodId
            }
          })

          res.json({ success: true })
        } catch (error) {
          res.status(400).json({ error: error.message })
        }
      })

      // Run once to sync existing plans to Stripe
      router.post('/sync-plans', async (req, res) => {
        try {
          const plans = await directus.items('service_package').readByQuery()

          for (const plan of plans.data) {
            if (!plan.stripe_price_id) {
              // Create Product
              const product = await stripe.products.create({
                name: plan.name,
                id: plan.id
              })

              // Create Price
              const price = await stripe.prices.create({
                product: product.id,
                unit_amount: plan.price_per_year,
                currency: 'aed',
                recurring: {
                  interval: 'year'
                }
              })

              // Update Directus
              await directus.items('service_package').updateOne(plan.id, {
                stripe_price_id: price.id
              })
            }
          }

          res.json({ success: true })
        } catch (error) {
          res.status(500).json({ error: error.message })
        }
      })

      router.post('/subscribe', async (req, res) => {
        const userService = new ItemsService('directus_users', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        const ServicePackages = new ItemsService('service_package', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        const Subscriptions = new ItemsService('subscription', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        try {
          let userId = req.accountability.user
          const user = await userService.readOne(userId)
          const { paymentMethodId, planId } = req.body
          const customerId = user.stripe_customer_id

          // 1. Attach payment method if not already attached
          const attachmentResult = await safelyAttachPaymentMethod(
            customerId,
            paymentMethodId
          )
          if (!attachmentResult.success) {
            return res.status(400).json({ error: attachmentResult.error })
          }

          // 2. Set as default payment method
          await stripe.customers.update(customerId, {
            invoice_settings: {
              default_payment_method: paymentMethodId
            }
          })

          // 3. Get plan details
          const plan = await ServicePackages.readOne(planId)
          let stripe_price_id = plan.stripe_price_id
          //3.1 if the plan is not in stripe uet then add it
          if (!stripe_price_id) {
            // Create Product
            const product = await stripe.products.create({
              name: plan.name,
              id: plan.id
            })

            // Create Price
            const price = await stripe.prices.create({
              product: product.id,
              unit_amount: plan.price_per_year * 100,
              currency: 'aed',
              recurring: {
                interval: 'year'
              }
            })

            // Update Directus
            await ServicePackages.updateOne(plan.id, {
              stripe_price_id: price.id
            })

            stripe_price_id = price.id
          }

          // 4. Create Stripe subscription
          // Create Stripe subscription with proper initial period
          const subscription = await stripe.subscriptions.create({
            customer: customerId,
            items: [{ price: stripe_price_id }],
            payment_behavior: 'default_incomplete', // For initial payment handling
            expand: ['latest_invoice.payment_intent'],
            billing_cycle_anchor: Math.floor(Date.now() / 1000) + 86400, // Current time + 1 day
            proration_behavior: 'none'
          })

          // Calculate initial period dates
          const now = new Date()
          const current_period_end = new Date(
            subscription.current_period_end * 1000
          )

          // Create local subscription with initialized values
          const subRecord = await Subscriptions.createOne({
            customer: req.accountability.user,
            package: planId,
            status: 'published', // Initial status
            stripe_sub_id: subscription.id,
            current_period_start: now,
            end_date: current_period_end,
            max_requests: plan.max_requests,
            cancel_at_period_end: req.body.cancel_at_period_end ?? false
          })
          res.json({
            success: true,
            subscription: subRecord,
            clientSecret:
              subscription.latest_invoice.payment_intent.client_secret
          })
        } catch (error) {
          return res.json({ error: error })
        }
      })

      // Configure in Stripe Dashboard: yourdomain.com/stripe-webhook
      router.post('/stripe-webhook', async (req, res) => {
        const Subscriptions = new ItemsService('subscription', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        const PaymentFailures = new ItemsService('payment_failures', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        const sig = req.headers['stripe-signature']
        let event

        try {
          event = stripe.webhooks.constructEvent(
            req.body,
            sig,
            STRIPE_WEBHOOK_SECRET
          )
        } catch (err) {
          return res.status(400).send(`Webhook Error: ${err.message}`)
        }
        const invoice = event.data.object
        const stripe_sub_id = invoice.subscription
        let subscription = await database('subscription')
          .where({ stripe_sub_id })
          .first()
        // Handle subscription events
        switch (event.type) {
          case 'invoice.paid':
            await Subscriptions.updateOne(subscription.id, {
              status: 'active',
              last_payment_at: new Date()
            })

            break

          case 'invoice.payment_failed':
            await Subscriptions.updateOne(subscription.id, {
              status: 'past_due',
              last_payment_failed_at: new Date()
            })
            // 6. Log the failure for analytics
            await PaymentFailures.createOne({
              user: customer.metadata.directus_user_id,
              stripe_sub_id: stripe_sub_id,
              invoice_id: invoice.id,
              amount: invoice.amount_due,
              failure_reason:
                invoice.payment_intent?.last_payment_error?.message ||
                'unknown',
              payment_method: invoice.payment_method
            })
            break

          case 'customer.subscription.updated':
            await handleSubscriptionUpdated(event.data.object)
            break
        }

        res.json({ received: true })
      })
    }

    // customer registration routes
    {
      function generateOtp () {
        const otp = Math.floor(100000 + Math.random() * 900000)
        return otp.toString()
      }

      router.post('/register', async (req, res) => {
        const { phone_number, first_name, last_name } = req.body

        try {
          const userService = new ItemsService('directus_users', {
            schema: await getSchema()
          })

          // Check if the phone number already exists
          const existingUser = await userService.readByQuery({
            filter: { phone_number },
            limit: 1
          })

          if (existingUser.length > 0) {
            if (existingUser[0].is_verified) {
              return res.status(400).json({
                success: false,
                message: 'Phone number already registered'
              })
            } else {
              // delete the existing user
              userService.deleteOne(existingUser[0].id)
            }
          }

          // Create new user
          const user = await userService.createOne({
            phone_number: phone_number,
            first_name: first_name,
            last_name: last_name,
            is_verified: false
          })

          res.json({ success: true, user: user })
        } catch (error) {
          res.status(500).json({
            success: false,
            error: error.message
          })
        }
      })

      router.post('/request-otp', async (req, res) => {
        const userService = new ItemsService('directus_users', {
          schema: await getSchema()
        })

        try {
          const { phone_number } = req.body

          // Find the user by phone number
          const user = await userService.readByQuery({
            filter: { phone_number },
            limit: 1
          })

          const otp_code = generateOtp() // Generate a random OTP

          var body = {
            Text: otp_code,
            Number: phone_number,
            // SenderId: 'askIITIans',
            SenderId: 'Raha App',
            Tool: 'API'
          }
          const authKey = '0OpYucxJHRkNQyUmNBjE'
          const authToken = 'DjoYKsHoJIJP2NEdJivK6cB5GuvqDReWMwQRtsRS'
          const credentials = Buffer.from(`${authKey}:${authToken}`).toString(
            'base64'
          )
          const response = await fetch(
            `https://restapi.smscountry.com/v0.1/Accounts/${authKey}/SMSes/`,
            {
              method: 'POST',
              headers: {
                Authorization: `Basic ${credentials}`,
                'Content-Type': 'application/json'
              },
              body: JSON.stringify(body)
            }
          )
          // this is not working now, send the otp anyway for testing purposes

          if (!response.ok) {
            const errorText = await response.text()
            res.json({
              errort: errorText
            })
          }
          if (user.length > 0) {
            await userService.updateOne(user[0].id, {
              otp_code: otp_code
            })
          }

          res.json({
            success: true,
            otp: otp_code,
            response: response,
            message: 'OTP sent. Please verify to complete the signup.'
          })
        } catch (error) {
          res.status(500).json({ success: false, message: error.message })
        }
      })

      router.post('/otplogin', async (req, res) => {
        const { phone_number, otp_code } = req.body

        try {
          const userService = new UsersService({
            schema: await getSchema(),
            accountability: req.accountability
          })
          // Find the user by phone number and OTP
          const user = await userService.readByQuery({
            filter: {
              phone_number,
              otp_code
            },
            limit: 1
          })

          if (user.length === 0) {
            return res
              .status(400)
              .json({ success: false, message: 'Invalid OTP or phone number' })
          }

          const payload = {
            userId: user[0].id, // Custom data
            iat: Math.floor(Date.now() / 1000), // Issued at time
            exp: Math.floor(Date.now() / 1000) + 60 * 60 * 3 // Expiration time (10000 hour from now)
          }

          const token = jwt.sign(payload, env.JWT_SECRET)

          await userService.updateOne(user[0].id, {
            token: token
          })
          // let updatedUser = await userService.readOne(user[0].id)
          // updatedUser['xtoken'] = token

          res.json({ success: true, ntoken: token })
        } catch (error) {
          res.status(500).json({
            success: false,
            message: 'Internal Server Error',
            errors: error.message
          })
        }
      })

      router.post('/refresh-token', async (req, res) => {
        const token = req.headers['authorization']

        if (!token) {
          return res
            .status(401)
            .json({ success: false, message: 'No token provided' })
        }

        try {
          const decoded = jwt.verify(token, env.JWT_SECRET)
          const userService = new ItemsService('directus_users', {
            schema: await getSchema()
          })

          const user = await userService.readOne(decoded.userId)

          if (!user) {
            return res
              .status(404)
              .json({ success: false, message: 'User not found' })
          }

          // Generate new JWT token
          const newToken = jwt.sign({ userId: user.id }, env.JWT_SECRET, {
            expiresIn: '10y'
          })

          res.json({ success: true, token: newToken })
        } catch (error) {
          console.error(error)
          res
            .status(500)
            .json({ success: false, message: 'Internal Server Error' })
        }
      })

      router.post('/logout', async (req, res) => {
        // Invalidate the session token
        // Since JWTs are stateless, on the client side, you would remove the token from storage (e.g., localStorage)
        res.json({ success: true, message: 'Logged out successfully' })
      })
    }

    // customer profile routes
    {
      router.post('/add_address', async (req, res) => {
        const Address = new ItemsService('address', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        const Emirates = new ItemsService('emirate', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        const userId = req.accountability.user

        try {
          let requiredFields = [
            'street_name',
            'building_or_house_number',
            'locality',
            'city',
            'emirate',
            'address_map'
          ]
          const allFieldsPresent = requiredFields.every(
            field => field in req.body
          )
          if (!allFieldsPresent) {
            // Throw a custom error with a status code and message
            const error = new Error('Missing required fields in request body.')
            error.statusCode = 400 // Bad Request
            throw error
          }
          let emirate = await Emirates.readByQuery({
            filter: {
              id: {
                _eq: req.body.emirate
              }
            }
          })
          if (emirate.length == 0) {
            const error = new Error('Wrong Emirate ID')
            error.statusCode = 400 // Bad Request
            throw error
          }

          let data = await Address.createOne({
            customer: userId,
            ...req.body
          })
          return res.json({ success: true, data: data })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })

      router.post('/update_address', async (req, res) => {
        const Address = new ItemsService('address', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        const Emirates = new ItemsService('emirate', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        const userId = req.accountability.user

        try {
          let requiredFields = [
            'street_name',
            'building_or_house_number',
            'locality',
            'city',
            'emirate',
            'address_map'
          ]
          const allFieldsPresent = requiredFields.every(
            field => field in req.body
          )
          if (!allFieldsPresent) {
            // Throw a custom error with a status code and message
            const error = new Error('Missing required fields in request body.')
            error.statusCode = 400 // Bad Request
            throw error
          }

          let address = await Address.readByQuery({
            filter: {
              id: {
                _eq: req.body.id
              },
              customer: {
                _eq: userId
              }
            }
          })
          if (address.length == 0) {
            const error = new Error('Wrong Address ID')
            error.statusCode = 400 // Bad Request
            throw error
          }

          let emirate = await Emirates.readByQuery({
            filter: {
              id: {
                _eq: req.body.emirate
              }
            }
          })
          if (emirate.length == 0) {
            const error = new Error('Wrong Emirate ID')
            error.statusCode = 400 // Bad Request
            throw error
          }

          await Address.updateOne(req.body.id, { ...req.body })

          return res.json({ success: true })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })

      router.post('/delete_address', async (req, res) => {
        const Address = new ItemsService('address', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        const ServiceRequests = new ItemsService('service_request', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        const userId = req.accountability.user

        const { id } = req.body

        try {
          let address = await Address.readByQuery({
            filter: {
              id: {
                _eq: id
              },
              customer: {
                _eq: userId
              }
            }
          })
          if (address.length == 0) {
            const error = new Error('Wrong Address Id')
            error.statusCode = 400 // Bad Request
            throw error
          }

          let pending_jobs = await ServiceRequests.readByQuery({
            filter: {
              customer: {
                _eq: userId
              },
              pick_up_address: {
                _eq: id
              },
              request_status: {
                _neq: 'completed'
              }
            }
          })
          if (pending_jobs.length > 0) {
            const error = new Error(
              'Address could not be deleted because it is used for at least one pending job.'
            )
            error.statusCode = 400 // Bad Request
            throw error
          }

          await Address.deleteOne(id)
          return res.json({ success: true })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })

      router.post('/add_car', async (req, res) => {
        const CustomerCars = new ItemsService('customer_car', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        const CustomerCarImages = new ItemsService('customer_car_files', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        const CustomerCarCards = new ItemsService('customer_car_files_1', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        const Emirates = new ItemsService('emirate', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        const CarModels = new ItemsService('car_model', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        const userId = req.accountability.user

        try {
          let requiredFields = [
            'model',
            'model_year',
            'emirate',
            'plate_number',
            'chassis_number'
          ]
          const allFieldsPresent = requiredFields.every(
            field => field in req.body
          )
          if (!allFieldsPresent) {
            // Throw a custom error with a status code and message
            const error = new Error('Missing required fields in request body.')
            error.statusCode = 400 // Bad Request
            throw error
          }

          let emirateId = req.body.emirate
          let modelId = req.body.model

          let emirate = await Emirates.readByQuery({
            filter: {
              id: {
                _eq: emirateId
              }
            }
          })
          if (emirate.length == 0) {
            const error = new Error('Wrong Emirate ID')
            error.statusCode = 400 // Bad Request
            throw error
          }

          let car_model = await CarModels.readByQuery({
            filter: {
              id: {
                _eq: modelId
              },
              status: {
                _eq: 'enabled'
              }
            }
          })
          if (car_model.length == 0) {
            const error = new Error('Wrong model ID')
            error.statusCode = 400 // Bad Request
            throw error
          }

          let data = await CustomerCars.createOne({
            customer: userId,
            ...req.body
          })

          for (const file of req.body.car_images) {
            await CustomerCarImages.createOne({
              customer_car_id: data,
              directus_files_id: file
            })
          }
          for (const file of req.body.cards_images) {
            await CustomerCarCards.createOne({
              customer_car_id: data,
              directus_files_id: file
            })
          }

          return res.json({ success: true, data: data })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })

      router.post('/update_car', async (req, res) => {
        const CustomerCars = new ItemsService('customer_car', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        const Emirates = new ItemsService('emirate', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        const CarModels = new ItemsService('car_model', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        const ServiceRequests = new ItemsService('service_request', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        const userId = req.accountability.user

        try {
          let requiredFields = [
            'id',
            'model',
            'model_year',
            'emirate',
            'plate_number',
            'chassis_number'
          ]
          const allFieldsPresent = requiredFields.every(
            field => field in req.body
          )
          if (!allFieldsPresent) {
            // Throw a custom error with a status code and message
            const error = new Error('Missing required fields in request body.')
            error.statusCode = 400 // Bad Request
            throw error
          }

          let carId = req.body.id
          let emirateId = req.body.emirate
          let modelId = req.body.model

          let emirate = await Emirates.readByQuery({
            filter: {
              id: {
                _eq: emirateId
              }
            }
          })
          if (emirate.length == 0) {
            const error = new Error('Wrong Emirate ID')
            error.statusCode = 400 // Bad Request
            throw error
          }

          let car_model = await CarModels.readByQuery({
            filter: {
              id: {
                _eq: modelId
              },
              status: {
                _eq: 'enabled'
              }
            }
          })
          if (car_model.length == 0) {
            const error = new Error('Wrong model ID')
            error.statusCode = 400 // Bad Request
            throw error
          }

          let customer_car = await CustomerCars.readOne(carId)
          if (customer_car.customer != userId) {
            const error = new Error('Wrong car ID')
            error.statusCode = 400 // Bad Request
            throw error
          }

          let service_request = ServiceRequests.readByQuery({
            filter: {
              car: {
                _eq: carId
              },
              status: {
                _neq: 'new'
              }
            }
          })
          if (service_request.length > 0) {
            const error = new Error(
              'Car could not be udpated, it has one or more service requests'
            )
            error.statusCode = 400 // Bad Request
            throw error
          }

          let data = await CustomerCars.createOne({
            customer: userId,
            ...req.body
          })
          return res.json({ success: true, data: data })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })

      router.post('/delete_car', async (req, res) => {
        const CustomerCars = new ItemsService('customer_car', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        const ServiceRequests = new ItemsService('service_request', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        const userId = req.accountability.user

        const { id } = req.body

        try {
          let car = await CustomerCars.readByQuery({
            filter: {
              id: {
                _eq: id
              },
              customer: {
                _eq: userId
              }
            }
          })
          if (car.length == 0) {
            const error = new Error('Wrong car Id')
            error.statusCode = 400 // Bad Request
            throw error
          }

          let service_request = await ServiceRequests.readByQuery({
            filter: {
              customer: {
                _eq: userId
              },
              car: {
                _eq: id
              }
            }
          })
          if (service_request.length > 0) {
            const error = new Error(
              'Car could not be deleted because it is used for at least one pending job.'
            )
            error.statusCode = 400 // Bad Request
            throw error
          }

          await CustomerCars.deleteOne(id)
          return res.json({ success: true })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })

      router.post('/xsubscribe', async (req, res) => {
        const ServicePackages = new ItemsService('service_package', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        const Subscriptions = new ItemsService('subscription', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        const userId = req.accountability.user

        const { id, auto_renew } = req.body

        try {
          let subscriptions = await Subscriptions.readByQuery({
            filter: {
              customer: {
                _eq: userId
              },
              end_date: {
                _gt: new Date()
              }
            }
          })

          if (subscriptions.length > 0) {
            const error = new Error('Customer has another active subscription')
            error.statusCode = 400 // Bad Request
            throw error
          }

          let servicePackage = await ServicePackages.readOne(id)

          if (servicePackage.status != 'enabled') {
            const error = new Error('The service package is not active')
            error.statusCode = 400 // Bad Request
            throw error
          }
          // Calculate current date and one year from now
          const currentDate = new Date()
          const oneYearFromNow = new Date()
          oneYearFromNow.setFullYear(currentDate.getFullYear() + 1)

          let subscription = await Subscriptions.createOne({
            customer: userId,
            package: id,
            status: 'draft',
            current_period_start: currentDate,
            end_date: oneYearFromNow,
            max_requests: servicePackage.max_requests,
            made_requests: 0,
            cancel_at_period_end: cancel_at_period_end
          })

          return res.json({ success: true, subscription: subscription })
        } catch (error) {
          return res.json({
            success: false,
            error: error.message
          })
        }
      })

      router.get('/profile', async (req, res) => {
        // const Customers = new UsersService('subscription', {
        //   schema: req.schema,
        //   accountability: req.accountability,
        //   headers: req.headers
        // })
        const RentingRequests = ItemsService('renting_request', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        const Subscriptions = new ItemsService('subscription', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        const CustomerCars = new ItemsService('customer_car', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        const Address = new ItemsService('address', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        const ServiceRequests = new ItemsService('service_request', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        const userId = req.accountability.user

        try {
          let customer = await UsersService.readOne(userId)

          let cars = await CustomerCars.readByQuery({
            filter: {
              customer: {
                _eq: userId
              },
              fields: ['*', 'model.*.*'],
              limit: -1
            }
          })
          let addresses = await Address.readByQuery({
            filter: {
              customer: {
                _eq: userId
              },
              fields: ['*', 'emirate.*'],
              limit: -1
            }
          })
          let requests = await ServiceRequests.readByQuery({
            filter: {
              customer: {
                _eq: userId
              },
              limit: -1
            }
          })
          let rentals = await RentingRequests.readByQuery({
            filter: {
              customer: {
                _eq: userId
              },
              fields: ['*', 'company_car.*.*'],
              limit: -1
            }
          })
          let subscription = await Subscriptions.readByQuery({
            filter: {
              customer: {
                _eq: userId
              },
              end_date: {
                _gte: new Date()
              },
              limit: 1
            }
          })

          return res.json({
            success: true,
            data: {
              customer: customer,
              addresses: addresses,
              cars: cars,
              request: requests,
              rentals: rentals,
              subscription: subscription.length == 1 ? subscription[0] : null
            }
          })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })
    }

    // service requests routes
    {
      router.post('/add_service_request', async (req, res) => {
        const ServiceRequests = new ItemsService('service_request', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        const CustomerCars = new ItemsService('customer_car', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        const Address = new ItemsService('address', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        const Workshop = new ItemsService('workshop', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        const CarServices = new ItemsService('car_service', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        const Subscriptions = new ItemsService('subscription', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        const ServicemanAssignments = new ItemsService(
          'serviceman_assignment',
          {
            schema: req.schema,
            accountability: req.accountability,
            headers: req.headers
          }
        )
        try {
          const userId = req.accountability.user
          let {
            car,
            fix_location,
            request_type,
            subscription,
            service,
            pick_up_address,
            pick_up_time,
            drop_off_address,
            drop_off_time,
            customer_comments,
            preferred_workshop
          } = req.body

          let requiredFields = [
            'car',
            'request_type',
            'fix_location',
            'pick_up_address'
          ]
          const allFieldsPresent = requiredFields.every(
            field => field in req.body
          )
          if (!allFieldsPresent) {
            const error = new Error('Missing required fields in request body.')
            error.statusCode = 400 // Bad Request
            throw error
          }
          if (fix_location === 'on-spot' && drop_off_address) {
            const error = new Error(
              'On spot fix does not allow drop off address.'
            )
            error.statusCode = 400 // Bad Request
            throw error
          }
          if (fix_location === 'on-spot' && preferred_workshop) {
            const error = new Error(
              'On spot fix does not allow selecting preferred workshop.'
            )
            error.statusCode = 400 // Bad Request
            throw error
          }
          if (request_type != 'on-demand' && request_type != 'subscription') {
            const error = new Error('Wrong request_type')
            error.statusCode = 400 // Bad Request
            throw error
          }
          if (request_type === 'on-demand' && !service) {
            const error = new Error('Missing service in request body.')
            error.statusCode = 400 // Bad Request
            throw error
          }
          if (request_type === 'subscription' && !subscription) {
            const error = new Error('Missing subscription in request body.')
            error.statusCode = 400 // Bad Request
            throw error
          }
          if (request_type === 'subscription' && service) {
            const error = new Error(
              'Wrong usage, request type subscription does not require service!'
            )
            error.statusCode = 400 // Bad Request
            throw error
          }
          if (request_type === 'on-demand' && subscription) {
            const error = new Error(
              'Wrong usage, request type on-demand does not require subscription!'
            )
            error.statusCode = 400 // Bad Request
            throw error
          }

          let qureyResponse

          if (service) {
            qureyResponse = await CarServices.readByQuery({
              filter: {
                id: {
                  _eq: service
                }
              }
            })
            if (qureyResponse.length === 0) {
              const error = new Error('Wrong service Id.')
              error.statusCode = 400 // Bad Request
              throw error
            }
            if (qureyResponse[0].status != 'enabled') {
              const error = new Error('Service is not enabled.')
              error.statusCode = 400 // Bad Request
              throw error
            }
          }
          if (subscription) {
            qureyResponse = await Subscriptions.readByQuery({
              filter: {
                id: {
                  _eq: subscription
                },
                customer: {
                  _eq: userId
                }
              }
            })
            if (qureyResponse.length === 0) {
              const error = new Error('Wrong subscription Id.')
              error.statusCode = 400 // Bad Request
              throw error
            }
            if (qureyResponse[0].end_date < new Date()) {
              const error = new Error('Subscription is expired.')
              error.statusCode = 400 // Bad Request
              throw error
            }
            let subscription = qureyResponse[0]
            qureyResponse = await ServiceRequests.readByQuery({
              filter: {
                subscription: {
                  _eq: subscription
                }
              }
            })
            if (qureyResponse.length > subscription.max_requests) {
              const error = new Error('Subscription is fully consumed.')
              error.statusCode = 400 // Bad Request
              throw error
            }
          }
          if (pick_up_address) {
            qureyResponse = await Address.readByQuery({
              filter: {
                id: {
                  _eq: pick_up_address
                },
                user: {
                  _eq: userId
                }
              }
            })
            if (qureyResponse.length === 0) {
              const error = new Error('Wrong pick_up_address Id.')
              error.statusCode = 400 // Bad Request
              throw error
            }
          }
          if (drop_off_address) {
            qureyResponse = await Address.readByQuery({
              filter: {
                id: {
                  _eq: drop_off_address
                },
                user: {
                  _eq: userId
                }
              }
            })
            if (qureyResponse.length === 0) {
              const error = new Error('Wrong drop_off_address Id.')
              error.statusCode = 400 // Bad Request
              throw error
            }
          } else {
            drop_off_address = pick_up_address
          }
          qureyResponse = await CustomerCars.readByQuery({
            filter: {
              id: {
                _eq: car
              },
              customer: {
                _eq: userId
              }
            }
          })
          if (qureyResponse.length === 0) {
            const error = new Error('Wrong car Id.')
            error.statusCode = 400 // Bad Request
            throw error
          }

          if (preferred_workshop) {
            qureyResponse = await Workshop.readByQuery({
              filter: {
                id: {
                  _eq: preferred_workshop
                }
              }
            })
            if (qureyResponse.length == 0) {
              const error = new Error('Wrong workshop Id')
              error.statusCode = 400 // Bad Request
              throw error
            }
          }

          if (pick_up_time && drop_off_time) {
            const pickUpDate = new Date(pick_up_time)
            const dropOffDate = new Date(drop_off_time)
            const currentDate = new Date()
            const sevenDaysLater = new Date(
              currentDate.getTime() + 7 * 24 * 60 * 60 * 1000
            ) // 7 days from now

            // Ensure pick_up_time is before drop_off_time
            if (pickUpDate >= dropOffDate) {
              const error = new Error(
                'Pick-up time must be earlier than drop-off time.'
              )
              error.statusCode = 400
              throw error
            }

            // Ensure both dates are within 7 days
            if (pickUpDate > sevenDaysLater || dropOffDate > sevenDaysLater) {
              const error = new Error(
                'Pick-up and drop-off times must be within 7 days.'
              )
              error.statusCode = 400
              throw error
            }
            // Ensure both dates are not in the past
            if (pickUpDate < currentDate || dropOffDate < currentDate) {
              const error = new Error(
                'Pick-up and drop-off times cannot be in the past.'
              )
              error.statusCode = 400
              throw error
            }
          }

          let service_request = await ServiceRequests.createOne({
            customer: userId,
            ...req.body
          })

          await ServicemanAssignments.createOne({
            job: service_request,
            status: 'pending'
          })

          return res.json({ success: true, service_request: service_request })
        } catch (error) {
          return res.json({
            success: false,
            message: error.message,
            error: error
          })
        }
      })

      router.post('/delete_request', async (req, res) => {
        const ServiceRequests = new ItemsService('service_request', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        try {
          const userId = req.accountability.user
          const { id } = req.body

          if (!id) {
            const error = new Error('missing request Id (id).')
            error.statusCode = 400 // Bad Request
            throw error
          }
          let qureyResponse = await ServiceRequests.readByQuery({
            filter: {
              id: {
                _eq: id
              }
            }
          })
          if (qureyResponse.length === 0) {
            const error = new Error('wrong request Id.')
            error.statusCode = 400 // Bad Request
            throw error
          }
          if (qureyResponse[0].customer != userId) {
            const error = new Error('wrong request Id.')
            error.statusCode = 400 // Bad Request
            throw error
          }
          if (qureyResponse[0].request_status != 'new') {
            const error = new Error('Request could not be deleted')
            error.statusCode = 400 // Bad Request
            throw error
          }

          await ServiceRequests.deleteOne(id)

          return res.json({ success: true })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })

      router.post('/add_renting_request', async (req, res) => {
        const CompanyCar = new ItemsService('company_car', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        const RentingRequest = new ItemsService('renting_request', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        const Address = new ItemsService('address', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        const userId = req.accountability.user

        try {
          let requiredFields = [
            'company_car',
            'from_date',
            'from_time',
            'to_date',
            'to_time'
          ]
          const allFieldsPresent = requiredFields.every(
            field => field in req.body
          )
          if (!allFieldsPresent) {
            // Throw a custom error with a status code and message
            const error = new Error('Missing required fields in request body.')
            error.statusCode = 400 // Bad Request
            throw error
          }

          const {
            company_car,
            from_date,
            to_date,
            from_time,
            to_time,
            pick_up_point,
            pick_up_remarks,
            drop_off_point,
            drop_off_remarks,
            pick_up_address,
            drop_off_address
          } = req.body

          let company_car_db = await CompanyCar.readByQuery({
            filter: {
              id: {
                _eq: company_car
              }
            }
          })

          if (company_car_db.length == 0) {
            const error = new Error('Wrong car ID')
            error.statusCode = 400 // Bad Request
            throw error
          }
          const pickUpDate = new Date(from_date)
          const dropOffDate = new Date(to_date)
          const currentDate = new Date()
          // Ensure pick_up_time is before drop_off_time
          if (pickUpDate >= dropOffDate)
            throw Object.assign(
              new Error('Pick-up time must be earlier than drop-off time.'),
              { statusCode: 400 }
            )

          // Ensure both dates are not in the past
          if (pickUpDate < currentDate || dropOffDate < currentDate)
            throw Object.assign(
              new Error('Pick-up and drop-off times cannot be in the past.'),
              { statusCode: 400 }
            )

          // either drop-off point or drop-off address should be defined
          if (!drop_off_address && !drop_off_point)
            throw Object.assign(
              new Error('drop off address and point cant be both undefined.'),
              { statusCode: 400 }
            )

          if (drop_off_address && drop_off_point)
            throw Object.assign(
              new Error('drop off address and point cant be both defined.'),
              { statusCode: 400 }
            )

          // either pick-up point or pick-up address should be defined
          if (!pick_up_address && !pick_up_point)
            throw Object.assign(
              new Error('drop off address and point cant be both undefined.'),
              { statusCode: 400 }
            )
          if (pick_up_address && pick_up_point)
            throw Object.assign(
              new Error('drop off address and point cant be both defined.'),
              { statusCode: 400 }
            )

          let renting_requests = await RentingRequest.readByQuery({
            filter: {
              company_car: {
                _eq: company_car
              },
              from_date: {
                _lte: to_date
              },
              to_date: {
                _gte: from_date
              },
              status: {
                _in: ['scheduled', 'started']
              }
            }
          })

          if (renting_requests.length > 0) {
            const error = new Error('Car is not available for this period')
            error.statusCode = 400 // Bad Request
            throw error
          }
          let number_of_days = getDaysBetween(from_date, to_date)
          let content = {
            ...req.body,
            customer: userId,
            number_of_days: number_of_days,
            total_charges: number_of_days * company_car_db[0].price_per_day ?? 0
          }
          let data = await RentingRequest.createOne(content)
          return res.json({ success: true, data: data })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })

      router.post('/pick_up_car', async (req, res) => {
        const CompanyCar = new ItemsService('company_car', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        const RentingRequest = new ItemsService('renting_request', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        const userId = req.accountability.user

        try {
          let renting_request_id = req.body.id
          let renting_request = await RentingRequest.readOne(renting_request_id)
          let company_car = await CompanyCar.readOne(
            renting_request.company_car
          )

          if (renting_request.status != 'scheduled') {
            const error = new Error('Renting Request is not Scheduled')
            error.statusCode = 400 // Bad Request
            throw error
          }

          const today = new Date()
          const todayStr = today.toISOString().split('T')[0] // 'yyyy-mm-dd'

          if (renting_request.from_date < todayStr) {
            const error = new Error(
              'Renting Request from date is not mature yet'
            )
            error.statusCode = 400 // Bad Request
            throw error
          }

          if (renting_request.to_date >= todayStr) {
            const error = new Error(
              'Renting Request to date is already passed!'
            )
            error.statusCode = 400 // Bad Request
            throw error
          }

          if (!company_car.can_be_rented) {
            const error = new Error('The company car is not for renting.')
            error.statusCode = 400
            throw error
          }

          if (!company_car.is_available) {
            const error = new Error(
              'The company car is not available now. Make sure previous renting is dropped off.'
            )
            error.statusCode = 400
            throw error
          }

          await CompanyCar.updateOne(company_car.id, { is_available: false })
          await RentingRequest.updateOne(renting_request.id, {
            status: 'started',
            pick_up_time: new Date(),
            pick_up_odo: company_car.current_odo_meter
          })

          return res.json({ success: true })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })

      router.post('/drop_off_car', async (req, res) => {
        const CompanyCar = new ItemsService('company_car', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        const RentingRequest = new ItemsService('renting_request', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        const userId = req.accountability.user

        try {
          let renting_request_id = req.body.id
          let renting_request = await RentingRequest.readOne(renting_request_id)
          let company_car = await CompanyCar.readOne(
            renting_request.company_car
          )

          if (renting_request.status != 'started') {
            const error = new Error('Renting Request is not Started')
            error.statusCode = 400 // Bad Request
            throw error
          }
          if (
            !renting_request.drop_off_odo ||
            renting_request.drop_off_odo <= 0
          ) {
            const error = new Error(
              'Drop off odometer reading should be recorded first.'
            )
            error.statusCode = 400 // Bad Request
            throw error
          }

          await CompanyCar.updateOne(company_car.id, {
            is_available: true,
            current_odo_meter: renting_request.drop_off_odo
          })
          await RentingRequest.updateOne(renting_request.id, {
            status: 'ended',
            drop_off_time: new Date()
          })

          return res.json({ success: true })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })
    }

    // Service man routes
    {
      router.get('/sm_service_request', async (req, res) => {
        const ServiceRequests = new ItemsService('service_request', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        const ServicemanLog = new ItemsService('serviceman_log', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        try {
          let service_request = await ServiceRequests.readOne(req.query.id, {
            fields: [
              '*',
              'car.*',
              'pick_up_address.*',
              'drop_off_address.*',
              'service.*',
              'customer.*'
            ]
          })
          let serviceman_log = await ServicemanLog.readByQuery({
            filter: {
              service_request: {
                _eq: req.query.id
              }
            }
          })

          return res.json({
            success: true,
            service_request: service_request,
            serviceman_log: serviceman_log
          })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })

      router.post('/sm_add_log', async (req, res) => {
        const ServiceRequests = new ItemsService('service_request', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        const ServicemanLog = new ItemsService('serviceman_log', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        const ServicemanLogFiles = new ItemsService('serviceman_log_files', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        const ServicemanAssignments = new ItemsService(
          'serviceman_assignment',
          {
            schema: req.schema,
            accountability: req.accountability,
            headers: req.headers
          }
        )
        try {
          let userId = req.accountability.user
          const {
            service_request_id,
            action_type,
            details,
            image_id,
            location,
            files
          } = req.body

          let requiredFields = ['service_request_id', 'action_type']
          const allFieldsPresent = requiredFields.every(
            field => field in req.body
          )
          if (!allFieldsPresent) {
            const error = new Error('Missing required fields in request body.')
            error.statusCode = 400 // Bad Request
            throw error
          }

          let qureyResponse = await ServiceRequests.readByQuery({
            filter: {
              id: {
                _eq: service_request_id
              }
            }
          })
          if (qureyResponse.length === 0) {
            const error = new Error('Wrong request id.')
            error.statusCode = 400 // Bad Request
            throw error
          }

          let service_request = qureyResponse[0]

          if (service_request.request_status === 'completed') {
            const error = new Error('Request is completed.')
            error.statusCode = 400 // Bad Request
            throw error
          }
          const assignment = await ServicemanAssignments.readByQuery({
            filter: {
              job: {
                _eq: service_request_id
              },
              serviceman: {
                _eq: userId
              },
              status: {
                _eq: 'accepted'
              }
            }
          })

          if (assignment.length === 0) {
            const error = new Error(
              'Please make sure the job is accepted by you first.'
            )
            error.statusCode = 400 // Bad Request
            throw error
          }

          let new_log = await ServicemanLog.createOne({
            service_request: service_request_id,
            action_type: action_type,
            details: details,
            image: image_id,
            location: location
          })
          for (const file of files) {
            await ServicemanLogFiles.createOne({
              serviceman_log_id: new_log,
              directus_files_id: file
            })
          }

          return res.json({ success: true, new_log: new_log })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })

      router.get('/sm_assignments', async (req, res) => {
        const ServicemanAssignments = new ItemsService(
          'serviceman_assignment',
          {
            schema: req.schema,
            accountability: req.accountability,
            headers: req.headers
          }
        )
        try {
          let userId = req.accountability.user

          let qureyResponse = await ServicemanAssignments.readByQuery({
            filter: {
              serviceman: {
                _eq: userId
              }
            },
            fields: ['*', 'job.*']
          })

          return res.json({ success: true, data: qureyResponse })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })

      router.post('/sm_accept_job', async (req, res) => {
        const ServicemanAssignments = new ItemsService(
          'serviceman_assignment',
          {
            schema: req.schema,
            accountability: req.accountability,
            headers: req.headers
          }
        )
        try {
          await ServicemanAssignments.updateOne(req.body.id, {
            status: 'accepted'
          })
          let qureyResponse = await ServicemanAssignments.readByQuery({
            filter: {
              serviceman: {
                _eq: userId
              }
            },
            fields: ['*', 'job.*']
          })
          return res.json({ success: true, data: qureyResponse })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })

      router.post('/sm_reject_job', async (req, res) => {
        const ServicemanAssignments = new ItemsService(
          'serviceman_assignment',
          {
            schema: req.schema,
            accountability: req.accountability,
            headers: req.headers
          }
        )
        try {
          let job = ServicemanAssignments.readOne(req.body.id)
          await ServicemanAssignments.updateOne(req.body.id, {
            status: 'rejected'
          })
          let qureyResponse = await ServicemanAssignments.readByQuery({
            filter: {
              serviceman: {
                _eq: userId
              }
            },
            fields: ['*', 'job.*']
          })
          await ServicemanAssignments.createOne({
            job: job.id,
            status: 'pending'
          })
          return res.json({ success: true, data: qureyResponse })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })
    }

    // Lookups
    {
      // lookups
      router.get('/car_brand', async (req, res) => {
        const CarBrands = new ItemsService('car_brand', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        try {
          let data = await CarBrands.readByQuery({
            filter: {
              status: {
                _eq: 'enabled'
              }
            },
            limit: -1
          })
          return res.json({ success: true, data: data })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })

      router.get('/car_model', async (req, res) => {
        const CarModels = new ItemsService('car_model', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        try {
          let data = await CarModels.readByQuery({
            filter: {
              status: {
                _eq: 'enabled'
              }
            },
            fields: ['*', 'brand.*'],
            limit: -1
          })

          return res.json({ success: true, data: data })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })

      router.get('/brand_model', async (req, res) => {
        const CarModels = new ItemsService('car_model', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        try {
          let { brand } = req.query
          let data = await CarModels.readByQuery({
            filter: {
              status: {
                _eq: 'enabled'
              },
              brand: {
                _eq: brand
              }
            },
            limit: -1
          })

          return res.json({ success: true, data: data })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })

      router.get('/car_category', async (req, res) => {
        const CarCategories = new ItemsService('car_category', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        try {
          let data = await CarCategories.readByQuery({
            limit: -1
          })

          return res.json({ success: true, data: data })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })

      router.get('/available_for_renting', async (req, res) => {
        const CompanyCars = new ItemsService('company_car', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        try {
          let { category } = req.query
          let data = await CompanyCars.readByQuery({
            filter: {
              can_be_rented: {
                _eq: true
              },
              is_available: {
                _eq: true
              },
              category: {
                _gte: category ? category : 0
              },
              category: {
                _lte: category ? category : 1000
              }
            },
            limit: -1
          })

          return res.json({ success: true, data: data })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })

      router.get('/emirate', async (req, res) => {
        const Emirates = new ItemsService('emirate', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        try {
          let data = await Emirates.readByQuery({
            limit: -1
          })

          return res.json({ success: true, data: data })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })

      router.get('/service_package', async (req, res) => {
        const ServicePackages = new ItemsService('service_package', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        try {
          let data = await ServicePackages.readByQuery({
            filter: {
              status: {
                _eq: 'enabled'
              }
            },
            fields: ['*'],
            limit: -1
          })
          return res.json({ success: true, data: data })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })

      router.get('/home_search', async (req, res) => {
        try {
          let { search_string } = req.query
          if (!search_string) search_string = ''
          // search car service
          let query = `
          SELECT 
              s.*, 
              c.name AS category_name 
          FROM car_service s
          JOIN service_category c ON s.category = c.id
          WHERE 
              ('' = 'SEARCH_STRING' OR 'SEARCH_STRING' IS NULL 
              OR s.name LIKE CONCAT('%', 'SEARCH_STRING', '%')
              OR s.description LIKE CONCAT('%', 'SEARCH_STRING', '%')
              OR c.name LIKE CONCAT('%', 'SEARCH_STRING', '%')
              OR CAST(s.included_services AS CHAR) LIKE CONCAT('%', 'SEARCH_STRING', '%'));         
          `
          query = query.replaceAll('SEARCH_STRING', search_string)
          let result = await database.raw(query)
          let car_service = result.rows

          // search service package
          query = `
          SELECT 
              s.*
          FROM service_package s
          WHERE 
              ('' = 'SEARCH_STRING' OR 'SEARCH_STRING' IS NULL 
              OR s.name LIKE CONCAT('%', 'SEARCH_STRING', '%')
              OR s.description LIKE CONCAT('%', 'SEARCH_STRING', '%')
              OR CAST(s.included_items AS CHAR) LIKE CONCAT('%', 'SEARCH_STRING', '%'));         
          `
          query = query.replaceAll('SEARCH_STRING', search_string)

          result = await database.raw(query)

          let service_package = result.rows

          // search company car
          query = `
          SELECT 
              s.*, 
              m.name AS model_name ,
              c.name AS category_name,
              b.name AS brand_name
          FROM company_car s
          JOIN car_model m ON s.model = m.id
          JOIN car_brand b ON m.brand = b.id
          JOIN car_category c ON s.category = c.id
          WHERE
          s.can_be_rented = true
          AND s.is_available = true
          AND ('' = 'SEARCH_STRING' OR 'SEARCH_STRING' IS NULL 
              OR m.name LIKE CONCAT('%', 'SEARCH_STRING', '%')
              OR b.name LIKE CONCAT('%', 'SEARCH_STRING', '%')
              OR c.name LIKE CONCAT('%', 'SEARCH_STRING', '%'))  ;         
          `
          query = query.replaceAll('SEARCH_STRING', search_string)
          result = await database.raw(query)
          let company_car = result.rows

          return res.json({
            success: true,
            car_service: car_service,
            service_package: service_package,
            company_car: company_car
          })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })

      router.get('/car_search', async (req, res) => {
        try {
          let {
            category,
            transmission,
            fuel_type,
            from_price,
            to_price,
            from_date,
            to_date
          } = req.query
          let query = `
          SELECT 
              s.*, 
              m.name AS model_name ,
              c.name AS category_name,
              b.name AS brand_name
          FROM company_car s
          JOIN car_model m ON s.model = m.id
          JOIN car_brand b ON m.brand = b.id
          JOIN car_category c ON s.category = c.id
          WHERE
          s.can_be_rented = true
          AND s.is_available = true
              
          `
          if (category) {
            query += ` AND category = ${category} `
          }
          if (transmission) {
            query += ` AND transmission = '${transmission}' `
          }
          if (fuel_type) {
            query += ` AND fuel_type = '${fuel_type}' `
          }
          if (from_price) {
            query += ` AND price_per_day >= ${from_price} `
          }
          if (to_price) {
            query += ` AND price_per_day <= ${to_price} `
          }
          if (from_date && to_date) {
            query += ` AND s.id not in (
            SELECT company_car FROM renting_requests WHERE status IN ('scheduled', 'started')
            AND ( ${from_date} <= to_date  AND ${to_date} >= from_date );
            )`
          }
          let result = await database.raw(query)
          let available_cars = result.rows

          return res.json({
            success: true,
            available_cars: available_cars
          })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })

      router.get('/workshops', async (req, res) => {
        const workshops = new ItemsService('workshop', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        try {
          const { address } = req.query
          let data
          if (!address) {
            // customer address is not provided;
            // select all workshops
            data = await workshops.readByQuery({
              filter: {
                status: {
                  _eq: 'published'
                }
              },
              fields: ['*'],
              limit: -1
            })
          } else {
            // customer address is provided;
            // select workshops in the same Emirate sorted by their distance from the address
            let query = `
            SELECT *,  (6371 * ACOS(COS(RADIANS(:lat)) * COS(RADIANS(latitude)) * 
            COS(RADIANS(longitude) - RADIANS(:lng)) + 
            SIN(RADIANS(:lat)) * SIN(RADIANS(latitude)) )) AS distance
            FROM workshops
            WHERE status = 'published' AND emirate = $EMIRATE
            ORDER BY distance;
            `
            query = query.replace('$EMIRATE', address.emirate)
            let result = await database.raw(query)
            data = result.rows
          }
          return res.json({ success: true, data: data })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })

      router.get('/car_service', async (req, res) => {
        const CarServices = new ItemsService('car_service', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        try {
          let data = await CarServices.readByQuery({
            filter: {
              status: {
                _eq: 'enabled'
              }
            },
            fields: ['*', 'category.*'],
            limit: -1
          })

          return res.json({ success: true, data: data })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })

      router.get('/package_services', async (req, res) => {
        const packageServices = new ItemsService('package_services', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })
        try {
          let data = await packageServices.readByQuery({
            filter: {
              status: {
                _eq: 'published'
              },
              package: {
                _eq: req.query.package
              }
            },
            fields: ['*', 'service.*'],
            limit: -1
          })

          return res.json({ success: true, data: data })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })

      router.get('/service_category', async (req, res) => {
        const ServiceCategories = new ItemsService('service_category', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        try {
          let data = await ServiceCategories.readByQuery({
            filter: {
              status: {
                _eq: 'enabled'
              }
            },
            limit: -1
          })

          return res.json({ success: true, data: data })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })

      router.get('/service_category_service', async (req, res) => {
        const CategoryServices = new ItemsService('category_service', {
          schema: req.schema,
          accountability: req.accountability,
          headers: req.headers
        })

        try {
          let { category } = req.query
          let data = await CategoryServices.readByQuery({
            filter: {
              status: {
                _eq: 'enabled'
              },
              category: {
                _eq: category
              }
            },
            limit: -1
          })

          return res.json({ success: true, data: data })
        } catch (error) {
          return res.json({ success: false, error: error.message })
        }
      })
    }
  }
}
